<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class ProfesoresController extends Controller
{
    public function index(Request $request)
    {
        $annio = (int) $request->query('annio', date('Y'));
        $q = trim((string) $request->query('q', ''));

        $rows = DB::table('dm_user as u')
            ->leftJoin(
                DB::raw('(SELECT director_id, GROUP_CONCAT(nombre ORDER BY nombre SEPARATOR ", ") AS director_de FROM sweb_salones WHERE director_id IS NOT NULL AND director_id > 0 GROUP BY director_id) d'),
                'd.director_id',
                '=',
                'u.id'
            )
            ->select(
                'u.id',
                'u.nombres',
                'u.apellido1',
                'u.apellido2',
                'u.telefono1',
                'u.username',
                'u.photo',
                DB::raw('COALESCE(d.director_de, "") as director_de')
            )
            ->whereRaw("TRIM(REPLACE(COALESCE(u.roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('u.is_active', 1)
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('u.nombres', 'like', "%{$q}%")
                        ->orWhere('u.apellido1', 'like', "%{$q}%")
                        ->orWhere('u.apellido2', 'like', "%{$q}%")
                        ->orWhere('u.username', 'like', "%{$q}%")
                        ->orWhere('u.telefono1', 'like', "%{$q}%");
                });
            })
            ->orderBy('u.nombres')
            ->paginate(25)
            ->withQueryString();

        return view('admin.profesores.index', compact('rows', 'q', 'annio'));
    }

    public function show(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $teacher = DB::table('dm_user')
            ->select(
                'id',
                'nombres',
                'apellido1',
                'apellido2',
                'fecha_nac',
                'sexo',
                'documento',
                'direccion',
                'telefono1',
                'email',
                'username',
                'photo',
                'is_active',
                'observacion'
            )
            ->where('id', $id)
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->first();

        abort_unless($teacher, 404);

        $firstName = trim((string) $teacher->nombres);
        $parts = preg_split('/\s+/', $firstName);
        $primerNombre = $parts[0] ?? '';
        $segundoNombre = implode(' ', array_slice($parts, 1));

        $meta = $this->extractTeacherMeta((string) ($teacher->observacion ?? ''));
        $tipoDcto = $meta['tipo_dcto'] ?? 'Cédula de Ciudadanía';

        return view('admin.profesores.show', compact(
            'teacher',
            'annio',
            'primerNombre',
            'segundoNombre',
            'tipoDcto'
        ));
    }

    public function create(Request $request)
    {
        $annio = (int) $request->query('annio', date('Y'));

        return view('admin.profesores.create', compact('annio'));
    }

    public function store(Request $request)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $data = $request->validate([
            'primer_nombre' => ['required', 'string', 'max:50'],
            'segundo_nombre' => ['nullable', 'string', 'max:100'],
            'primer_apellido' => ['required', 'string', 'max:30'],
            'segundo_apellido' => ['nullable', 'string', 'max:30'],
            'fecha_nac' => ['nullable', 'date'],
            'sexo' => ['nullable', Rule::in(['Masculino', 'Femenino'])],
            'tipo_dcto' => ['nullable', 'string', 'max:60'],
            'documento' => ['required', 'string', 'max:20', Rule::unique('dm_user', 'documento')],
            'direccion' => ['nullable', 'string', 'max:150'],
            'telefono1' => ['nullable', 'string', 'max:100'],
            'email' => ['required', 'email', 'max:255', Rule::unique('dm_user', 'email')],
            'username' => ['required', 'string', 'max:255', Rule::unique('dm_user', 'username')],
            'password' => ['required', 'string', 'min:6', 'max:100'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $fullName = trim(($data['primer_nombre'] ?? '') . ' ' . ($data['segundo_nombre'] ?? ''));
        $meta = ['tipo_dcto' => trim((string) ($data['tipo_dcto'] ?? 'Cédula de Ciudadanía'))];

        DB::table('dm_user')->insert([
            'uuid' => (string) Str::uuid(),
            'nombres' => $fullName,
            'apellido1' => $data['primer_apellido'] ?? null,
            'apellido2' => $data['segundo_apellido'] ?? null,
            'fecha_nac' => $data['fecha_nac'] ?? null,
            'sexo' => $data['sexo'] ?? null,
            'documento' => $data['documento'],
            'direccion' => $data['direccion'] ?? null,
            'telefono1' => $data['telefono1'] ?? null,
            'email' => $data['email'],
            'username' => $data['username'],
            'password' => sha1((string) $data['password']),
            'salt' => '',
            'algorithm' => 'sha1',
            'roll' => 'docentes',
            'is_active' => (int) ($data['is_active'] ?? 1),
            'photo' => null,
            'observacion' => json_encode($meta, JSON_UNESCAPED_UNICODE),
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        return redirect()
            ->route('admin.profesores.index', ['annio' => $annio])
            ->with('status', 'Profesor creado correctamente.');
    }

    public function update(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $teacher = DB::table('dm_user')
            ->select('id', 'nombres', 'observacion')
            ->where('id', $id)
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->first();

        abort_unless($teacher, 404);

        $data = $request->validate([
            'primer_nombre' => ['required', 'string', 'max:50'],
            'segundo_nombre' => ['nullable', 'string', 'max:100'],
            'primer_apellido' => ['required', 'string', 'max:30'],
            'segundo_apellido' => ['nullable', 'string', 'max:30'],
            'fecha_nac' => ['nullable', 'date'],
            'sexo' => ['nullable', Rule::in(['Masculino', 'Femenino'])],
            'tipo_dcto' => ['nullable', 'string', 'max:60'],
            'documento' => [
                'required',
                'string',
                'max:20',
                Rule::unique('dm_user', 'documento')->ignore($id),
            ],
            'direccion' => ['nullable', 'string', 'max:150'],
            'telefono1' => ['nullable', 'string', 'max:100'],
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('dm_user', 'email')->ignore($id),
            ],
            'username' => [
                'required',
                'string',
                'max:255',
                Rule::unique('dm_user', 'username')->ignore($id),
            ],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $fullName = trim(($data['primer_nombre'] ?? '') . ' ' . ($data['segundo_nombre'] ?? ''));
        if ($fullName === '') {
            $fullName = trim((string) $teacher->nombres);
        }

        $meta = $this->extractTeacherMeta((string) ($teacher->observacion ?? ''));
        $meta['tipo_dcto'] = trim((string) ($data['tipo_dcto'] ?? ''));
        $observacion = json_encode($meta, JSON_UNESCAPED_UNICODE);

        DB::table('dm_user')->where('id', $id)->update([
            'nombres' => $fullName,
            'apellido1' => $data['primer_apellido'] ?? null,
            'apellido2' => $data['segundo_apellido'] ?? null,
            'fecha_nac' => $data['fecha_nac'] ?? null,
            'sexo' => $data['sexo'] ?? null,
            'documento' => $data['documento'],
            'direccion' => $data['direccion'] ?? null,
            'telefono1' => $data['telefono1'] ?? null,
            'email' => $data['email'],
            'username' => $data['username'],
            'is_active' => (int) ($data['is_active'] ?? 0),
            'observacion' => $observacion,
            'updated_at' => Carbon::now(),
        ]);

        return redirect()
            ->route('admin.profesores.show', ['id' => $id, 'annio' => $annio])
            ->with('status', 'Profesor actualizado correctamente.');
    }

    private function extractTeacherMeta(string $observacion): array
    {
        if ($observacion === '') {
            return [];
        }

        $decoded = json_decode($observacion, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return $decoded;
        }

        return [
            '_legacy_note' => $observacion,
        ];
    }
}
